/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: PhysicalInterface.h 6424 2014-07-01 12:15:23Z stephans $
   Last Modification:
    $Author: stephans $
    $Date: 2014-07-01 14:15:23 +0200 (Di, 01 Jul 2014) $
    $Revision: 6424 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
    Declaration of the physical interface class
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         xx                   created
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file PhysicalInterface.h
/// Declaration of the cifX API Marshaller class interface
/////////////////////////////////////////////////////////////////////////////

#pragma once


class CTransportLayer;
class CDeviceHandler;

class CPhysicalInterface;
class CPhysicalLayer;

#include <stdint.h>
#include <string>
#include <map>
#include <crtdbg.h>
//#include "DeviceHandler.h"
#include "HilTransport.h"
//#include "TransportLayer.h"
#include "ConnectorAPI.h"
#include "netXConnectorErrors.h"
#include "cifXErrors.h"

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CPhysicalInterface
/// Single instance object which connects Transport layer and the physical drivers
///////////////////////////////////////////////////////////////////////////////////////////
class CPhysicalInterface
{
public:
  //////////////////////////////////////////////////
  /// Constructor
  ///   \param pcConnectorInterface   Pointer reference to interface object
  ///   \param ptConnector            Connector information structure
  ///   \param pcLayer                Pointer reference to physical layer object
  ///   \param szInterfaceName        Name of the interface
  //////////////////////////////////////////////////
  CPhysicalInterface( void* pcConnectorInterface, NETX_CONNECTOR_DLL_INFO_T* ptConnector, CPhysicalLayer* pcLayer, std::string& szInterfaceName)
  : m_pcTransport(NULL)
  , m_pvConnectorInterface( pcConnectorInterface)
  , m_ptConnector(ptConnector)
  , m_pcPhysicalLayer(pcLayer)
  , m_szInterfaceName(szInterfaceName)
  , m_hReconnect(NULL)
  {
    InitializeCriticalSection(&m_tLock);
  }

  // Destructor
  ~CPhysicalInterface()
  {
    DeleteCriticalSection(&m_tLock);
  }

  int32_t  Start               ( void);                                  /*!< Start the corresponding interface (connector)                */
  
  uint32_t GetState            ( void);
  
  uint32_t GetResetTimeout     ( void);
  
  uint32_t GetKeepAliveTimeout ( void);
  
  uint32_t GetSendTimeout      ( void);

  //////////////////////////////////////////////////
  /// Stop corresponding interface (connector)
  ///   \return   void
  //////////////////////////////////////////////////  
  void    Stop( void)                           
  { 
    m_ptConnector->tFncTable.pfnNetXConIntfStop( m_pvConnectorInterface); 
  }
  
  //////////////////////////////////////////////////
  /// Send data to physical interface (connector interface)
  ///   \param    pabData     Reference to the data
  ///   \param    ulDataLen   Length of data to send
  ///   \return   NXCON_NO_ERROR on success
  //////////////////////////////////////////////////
  int32_t Send( uint8_t* pabData,         
                uint32_t ulDataLen)  
  { 
    return m_ptConnector->tFncTable.pfnNetXConIntfSend( m_pvConnectorInterface,
                                                        pabData, 
                                                        ulDataLen);
  }

  //////////////////////////////////////////////////
  /// Set TransportLayer
  ///   \param    pcTransport Transport layer object 
  ///   \return   void
  //////////////////////////////////////////////////
  void SetTransportLayer(CTransportLayer* pcTransport)
  {
    m_pcTransport = pcTransport;
  }

  //////////////////////////////////////////////////
  /// GetTransportLayer
  ///   \return CTransportLayer object 
  //////////////////////////////////////////////////
  CTransportLayer* GetTransportLayer(void)
  {
    return m_pcTransport;
  }

  //////////////////////////////////////////////////
  /// GetConnectorInterface, returns the given interface pointer from the connector
  ///   \return m_pvConnectorInterface
  //////////////////////////////////////////////////
  void* GetConnectorInterface(void)
  {
    return m_pvConnectorInterface;
  }

  CPhysicalLayer* GetPhysicalLayer(void)
  {
    return m_pcPhysicalLayer;
  }

  void Lock(void)   {EnterCriticalSection(&m_tLock);}
  void Unlock(void) {LeaveCriticalSection(&m_tLock);}

  void  GetInterfaceName(std::string& szInterface)  { szInterface = m_szInterfaceName; }
  std::string&  GetInterfaceName(void)  { return m_szInterfaceName; }

  HANDLE                      m_hReconnect;

protected:
  std::string                 m_szInterfaceName;
  NETX_CONNECTOR_DLL_INFO_T*  m_ptConnector;
  void*                       m_pvConnectorInterface;                               
  CPhysicalLayer*             m_pcPhysicalLayer;
  CTransportLayer*            m_pcTransport;
  CRITICAL_SECTION            m_tLock;
};



///////////////////////////////////////////////////////////////////////////////////////////
/// \class CPhysicalLayer
///////////////////////////////////////////////////////////////////////////////////////////
class CPhysicalLayer
{
public:
//////////////////////////////////////////////////
/// Constructor
///   \param ptConnectorInfo  Connector of physical layer
//////////////////////////////////////////////////
  CPhysicalLayer( NETX_CONNECTOR_DLL_INFO_T *ptConnectorInfo) 
  : m_pcDeviceHandler(NULL)
  , m_ptConnector(ptConnectorInfo) 
  {
  }
  
  NETX_CONNECTOR_DLL_INFO_T* GetConnectorInfo(void) 
  { 
    return m_ptConnector;
  }
  
  //////////////////////////////////////////////////
  /// SetDeviceHandler, save device handler
  ///   \param  pcDeviceHandler
  //////////////////////////////////////////////////
  void SetDeviceHandler(CDeviceHandler* pcDeviceHandler)
  {
    m_pcDeviceHandler = pcDeviceHandler;
  }

  //////////////////////////////////////////////////
  /// GetDeviceHandler, saved device handler
  ///   \return Pointer reference to device handler object
  //////////////////////////////////////////////////
  CDeviceHandler* GetDeviceHandler( void)
  {
    return m_pcDeviceHandler;
  }


  //////////////////////////////////////////////////
  /// CreateInterface, creates a new CPhysicalInterface object
  ///   \param  szDeviceName      Name of the device (interface)
  ///   \return CPhysicalInterface reference
  //////////////////////////////////////////////////
  CPhysicalInterface* CreateInterface(std::string szDeviceName)
  {
    CPhysicalInterface* pcPysIF = NULL;

    void* pcConnectorInterface = (void*)m_ptConnector->tFncTable.pfnNetXConCreateInterface( szDeviceName.c_str());
    if (NULL != pcConnectorInterface)
    {
      /* Create a new interface object */
      pcPysIF = new CPhysicalInterface( pcConnectorInterface, m_ptConnector, this, szDeviceName);
    }
    return pcPysIF;
  }

  CDeviceHandler*     m_pcDeviceHandler;                                          /*!< Device handler object to handle notification callbacks       */

  int32_t             Init(void);                                                 /*!< Initialize the specific interface                            */

  void                Deinit(void);                                               /*!< DeInitialize the specific interface                          */
  
  HWND                CreateConfigDialog( HWND        hParentWnd, 
                                          const char* szConfig);                  /*!< Call physical layer to create a dialog in the given window   */
  
  void                EndDialog( void);                                           /*!< Call physical layer to close the created dialog              */ 

  int32_t             GetShortInterfaceName ( uint32_t ulSize, 
                                              char*         pszShortName, 
                                              char*         pszLongName);

  int32_t             GetLongInterfaceName  ( uint32_t ulSize, 
                                              char*         pszLongName, 
                                              char*         pszShortName);
  
  uint32_t            GetType               ( void);
  
  int32_t             GetConfig( NETX_CONNECTOR_CONFIG_CMD_E  eCmd, 
                                 void*                        pvConfig);          /*!< Get configuration of interface                               */

  int32_t             SetConfig( NETX_CONNECTOR_CONFIG_CMD_E  eCmd,
                                 const char* szConfig);                           /*!< Set configuration of interface                               */  

protected:
  NETX_CONNECTOR_DLL_INFO_T*  m_ptConnector;                                      /*!< Information struct of connector                              */

};


